﻿#include "precompiled.h"
#include "common.h"
#include "Entity.h"

#include "ResourceManager.h"
#include "Camera.h"
#include "Lens.h"
#include "Aperture.h"
#include "Film.h"
#include "Transform.h"
#include "MeshRenderer.h"

namespace RTCam {

Entity::Entity(const string& name) :
	m_name(name),
	m_modelCBufferData(),
	m_selfPtr(),
	m_parent(),
	m_children(),
	// Components
	m_transform(nullptr),
	m_camera(nullptr),
	m_renderer(nullptr)
{
}

Entity::~Entity(void)
{
}

shared_ptr<Entity> Entity::Create(const string& name)
{
	shared_ptr<Entity> newEntity(new Entity(name));
	newEntity->m_selfPtr = newEntity;
	newEntity->m_transform = shared_ptr<Transform>(new Transform(newEntity));
	return newEntity;
}

void Entity::AddChild( const shared_ptr<Entity>& child )
{
	ASSERT_MSG(child != nullptr, "Trying to add a shared pointer to null as a child entity.\n");
	ASSERT_MSG(child.get() != m_selfPtr.lock().get(), "An entity cannot be a child of itself.\n");

	m_children.push_back(child);
	child->m_parent = m_selfPtr.lock();
}

const shared_ptr<Transform>& Entity::GetTransform()
{
	return m_transform;
}

const shared_ptr<Camera>& Entity::GetCamera()
{
	return m_camera;
}

const shared_ptr<MeshRenderer>& Entity::GetRenderer()
{
	return m_renderer;
}

void Entity::AddCamera(const shared_ptr<ResourceManager>& resources)
{
	ASSERT(m_camera == nullptr);
	m_camera = shared_ptr<Camera>(new Camera(m_selfPtr.lock()));

	// Assign default textures
	m_camera->m_aperture->m_bokehTex = resources->m_texBokeh;
	
	m_camera->m_lens->m_positiveSATex = resources->m_texPositiveSA;
	m_camera->m_lens->m_zeroSATex = resources->m_texZeroSA;
	m_camera->m_lens->m_negativeSATex = resources->m_texNegativeSA;

	m_camera->m_lens->m_distortionTex = resources->m_texDefault1D;
	m_camera->m_lens->m_sagittalTex = resources->m_texDefault1D;
	m_camera->m_lens->m_tangentialTex = resources->m_texDefault1D;
	
	Lens::Vignetting v = {0};
	v.fNumber = 8;
	v.vignettingTex = resources->m_texDefaultVignetting;
	m_camera->m_lens->m_vignettingTexs.push_back(v); 

	m_camera->m_film->m_responseCurveTex = resources->m_texDefault1D;
}

void Entity::AddMeshRenderer(const shared_ptr<Mesh>& mesh)
{
	ASSERT(m_renderer == nullptr);
	m_renderer = shared_ptr<MeshRenderer>(new MeshRenderer(m_selfPtr.lock()));
	m_renderer->SetMesh(mesh);
}

shared_ptr<Entity> Entity::GetParent()
{
	return m_parent.lock();
}

const vector<shared_ptr<Entity>>& Entity::GetChildren()
{
	return m_children;
}

} // end namespace